document.addEventListener("DOMContentLoaded", async () => {
  const cardsContainer = document.getElementById("cards-container");
  const btnBarajar = document.getElementById("btn-barajar");
  const puzzles = [];
  const alturaPieza = 300;
  const margen = 32;
  const maxAncho = document.body.clientWidth - margen;

  try {
    // Usar Promise.all si tienes múltiples tarjetas
    for (const [index, linea] of textLines.entries()) {
      const { card, piezas, puzzleContainer, scrollWrapper } = await crearTarjeta(linea, index, maxAncho, alturaPieza);
      cardsContainer.appendChild(card);
      puzzles.push({ piezas, container: puzzleContainer });
      configurarScrollYDrag(scrollWrapper, puzzleContainer);
    }

    configurarDragAndDrop(cardsContainer);
    configurarTouchDrag(cardsContainer);

    btnBarajar.addEventListener("click", () => {
      barajarTodo(puzzles);
    });
  } catch (error) {
    console.error("Error al crear el rompecabezas:", error);
  }
});

// Crear la tarjeta con el rompecabezas
function crearTarjeta(linea, index, maxAncho, alturaPieza) {
  // Crear elemento principal de la tarjeta
  const card = document.createElement("div");
  card.className = "card mb-3 p-3";
  card.style.maxWidth = "100%";
  card.style.overflow = "hidden";

  // Añadir título (frase en inglés)
  const titulo = document.createElement("h5");
  titulo.className = "card-title";
  titulo.textContent = `Frase ${index + 1}: ${linea.en}`;
  card.appendChild(titulo);

  // Añadir traducción (frase en español)
  const traduccion = document.createElement("p");
  traduccion.className = "card-text text-muted mb-3";
  traduccion.textContent = linea.es;
  card.appendChild(traduccion);

  // Mostrar imagen completa de referencia
  const imgCompleta = document.createElement("img");
  imgCompleta.src = `${linea.img}?nocache=${Date.now()}`;
  imgCompleta.className = "img-fluid mb-3";
  imgCompleta.style.maxWidth = "100%";
  imgCompleta.style.height = "auto";
  imgCompleta.style.borderRadius = "4px";
  card.appendChild(imgCompleta);

  // Dividir la frase en palabras
  const palabras = linea.en.trim().split(/\s+/);
  const columnas = palabras.length;

  // Crear contenedor de scroll horizontal
  const scrollWrapper = document.createElement("div");
  scrollWrapper.className = "scroll-wrapper";
  scrollWrapper.style.width = "100%";
  scrollWrapper.style.overflowX = "auto";
  scrollWrapper.style.margin = "auto";
  scrollWrapper.style.userSelect = "none";
  scrollWrapper.style.webkitOverflowScrolling = "touch";
  scrollWrapper.style.touchAction = "pan-x";

  // Crear contenedor del rompecabezas (sin espacios)
  const puzzleContainer = document.createElement("div");
  puzzleContainer.className = "puzzle-container d-flex";
  puzzleContainer.style.height = `${alturaPieza}px`;
  puzzleContainer.style.marginBottom = "20px";
  puzzleContainer.style.touchAction = "none";
  puzzleContainer.style.userSelect = "none";
  puzzleContainer.style.gap = "0"; // Eliminar espacios
  puzzleContainer.style.padding = "0"; // Eliminar padding

  // Calcular ancho mínimo por pieza (80px o el ancho disponible)
  const anchoMinimoPieza = 80;
  const anchoDisponible = card.clientWidth;
  const anchoPieza = Math.max(anchoDisponible / columnas, anchoMinimoPieza);
  
  // Ajustar el ancho total del contenedor de piezas
  puzzleContainer.style.width = `${anchoPieza * columnas}px`;

  // Crear las piezas del rompecabezas (versión síncrona)
  const piezas = [];
  for (let x = 0; x < columnas; x++) {
    const pieza = document.createElement("div");
    pieza.className = "piece position-relative";
    pieza.style.width = `${anchoPieza}px`;
    pieza.style.height = `${alturaPieza}px`;
    pieza.style.flex = "0 0 auto"; // Fuerza el ancho exacto
    pieza.style.overflow = "hidden";
    pieza.style.cursor = "grab";
    pieza.style.borderRight = "1px solid #ccc"; // Solo borde derecho

    // Configurar imagen
    const img = document.createElement("img");
    img.src = `${linea.img}?nocache=${Date.now()}`;
    img.style.position = "absolute";
    img.style.height = "100%";
    img.style.width = "auto";
    img.style.pointerEvents = "none";
    img.style.userSelect = "none";
    
    img.onload = function() {
      const imgNaturalWidth = this.naturalWidth;
      const imgNaturalHeight = this.naturalHeight;
      const escala = alturaPieza / imgNaturalHeight;
      const imgDisplayWidth = imgNaturalWidth * escala;
      
      this.style.width = `${imgDisplayWidth}px`;
      this.style.left = `-${x * anchoPieza}px`;
    };

    pieza.appendChild(img);

    // Añadir texto vertical
    const palabra = palabras[x];
    const textoContenedor = document.createElement("div");
    textoContenedor.className = "pieza-texto-vertical";
    
    // Crear una letra por línea
    for (const letra of palabra) {
      const letraSpan = document.createElement("span");
      letraSpan.textContent = letra;
      textoContenedor.appendChild(letraSpan);
    }

    pieza.appendChild(textoContenedor);

    pieza.dataset.correct = x;
    pieza.draggable = true;
    pieza.dataset.index = x;
    
    piezas.push(pieza);
    puzzleContainer.appendChild(pieza);
  }

  // Eliminar borde derecho de la última pieza
  if (piezas.length > 0) {
    piezas[piezas.length - 1].style.borderRight = "none";
  }

  // Añadir elementos al DOM
  scrollWrapper.appendChild(puzzleContainer);
  card.appendChild(scrollWrapper);

  // Configurar eventos de scroll y drag
  configurarScrollYDrag(scrollWrapper, puzzleContainer);

  return { 
    card, 
    piezas, 
    puzzleContainer, 
    scrollWrapper 
  };
}

// Crear piezas con imagen responsive y texto encima
function crearPiezas(linea, columnas, contenedor, altura, anchoTotal) {
  const piezas = [];
  const anchoPieza = anchoTotal / columnas;
  
  // Ajustar el contenedor para que no añada espacios
  contenedor.style.gap = '0';
  contenedor.style.margin = '0';
  contenedor.style.padding = '0';

  for (let x = 0; x < columnas; x++) {
    const pieza = document.createElement("div");
    pieza.className = "piece";
    pieza.style.width = `${anchoPieza}px`; // Ancho exacto
    pieza.style.height = `${altura}px`;
    pieza.style.flex = '0 0 auto'; // Fuerza el ancho exacto

    // Configuración de la imagen...
    const img = document.createElement("img");
    img.src = `${linea.img}?nocache=${Date.now()}`;
    img.style.width = `${anchoTotal}px`; // Ancho de la imagen completa
    img.style.height = '100%';
    img.style.position = 'absolute';
    img.style.left = `-${x * anchoPieza}px`; // Desplazamiento preciso
    
    pieza.appendChild(img);

    // Añadir texto si es necesario...
    const textoSpan = document.createElement("span");
    textoSpan.textContent = linea.en.trim().split(/\s+/)[x];
    textoSpan.className = "pieza-texto";
    pieza.appendChild(textoSpan);

    pieza.dataset.correct = x;
    pieza.draggable = true;
    piezas.push(pieza);
    contenedor.appendChild(pieza);
  }

  return piezas;
}

// Bloquear scroll horizontal del contenedor durante drag/touch en piezas
function configurarScrollYDrag(scrollWrapper, puzzleContainer) {
  let isDragging = false;
  let touchDragging = false;

  puzzleContainer.addEventListener("dragstart", () => {
    isDragging = true;
  });

  puzzleContainer.addEventListener("dragend", () => {
    isDragging = false;
  });

  scrollWrapper.addEventListener("scroll", e => {
    if (isDragging) e.preventDefault();
  });


puzzleContainer.addEventListener("touchstart", e => {
  const target = e.target.closest(".piece");
  if (target) {
    touchDragging = true;
  } else {
    touchDragging = false; // importante limpiar si no es pieza
  }
});

puzzleContainer.addEventListener("touchend", () => {
  touchDragging = false;
});

scrollWrapper.addEventListener("touchmove", e => {
  if (touchDragging) {
    e.preventDefault(); // bloquear scroll sólo si arrastrando pieza
  }
}, { passive: false });



}

// Drag & Drop escritorio
function configurarDragAndDrop(container) {
  let piezaArrastrada = null;

  container.addEventListener("dragstart", e => {
    if (e.target.classList.contains("piece")) {
      piezaArrastrada = e.target;
      e.dataTransfer.effectAllowed = "move";
    }
  });

  container.addEventListener("dragover", e => {
    if (e.target.classList.contains("piece")) {
      e.preventDefault();
      e.dataTransfer.dropEffect = "move";
    }
  });

  container.addEventListener("drop", e => {
    if (e.target.classList.contains("piece") && piezaArrastrada) {
      e.preventDefault();

      const origen = piezaArrastrada;
      const destino = e.target;

      if (origen.parentNode !== destino.parentNode) return;

      const containerPiezas = origen.parentNode;
      intercambiarPiezas(origen, destino, containerPiezas);
    }
  });
}

// Touch drag móvil
function configurarTouchDrag(container) {
  let piezaTocada = null;

  container.addEventListener("touchstart", e => {
    const target = e.target.closest(".piece");
    if (target) piezaTocada = target;
  });

  container.addEventListener("touchend", e => {
    if (!piezaTocada) return;

    const touch = e.changedTouches[0];
    const piezaDestino = document.elementFromPoint(touch.clientX, touch.clientY)?.closest(".piece");

    if (
      piezaDestino &&
      piezaDestino !== piezaTocada &&
      piezaDestino.parentNode === piezaTocada.parentNode
    ) {
      intercambiarPiezas(piezaTocada, piezaDestino, piezaTocada.parentNode);
    }

    piezaTocada = null;
  });
}

// Intercambiar piezas en el DOM
function intercambiarPiezas(origen, destino, container) {
  const origenIndex = origen.dataset.index;
  const destinoIndex = destino.dataset.index;

  const origenClon = origen.cloneNode(true);
  const destinoClon = destino.cloneNode(true);

  origenClon.dataset.index = destinoIndex;
  destinoClon.dataset.index = origenIndex;

  container.replaceChild(destinoClon, origen);
  container.replaceChild(origenClon, destino);

  if (rompecabezasCompleto(container)) {
    console.log("¡Felicidades! Completaste el rompecabezas.");
  }
}

// Comprobar si puzzle está completo
function rompecabezasCompleto(container) {
  for (const pieza of container.children) {
    if (pieza.dataset.index !== pieza.dataset.correct) return false;
  }
  return true;
}

// Barajar todas las piezas
function barajarTodo(puzzles) {
  puzzles.forEach(({ piezas, container }) => {
    piezas.sort(() => Math.random() - 0.5);
    container.innerHTML = "";
    piezas.forEach((pieza, i) => {
      pieza.dataset.index = i;
      container.appendChild(pieza);
    });
  });
}
